/*---------------------------------------------------------------------------*\

    FILE....: TLOADBEEP.CPP
    TYPE....: C++ Program
    AUTHOR..: David Rowe
    DATE....: 8/6/01

    This program tests the V4PCI by loading it with 4 simultaneous
    channels of audio I/O.  Unlike tload.cpp, this program provides its own
    source of audio, by looping back the channels.  A PBX is not required.

    1. Connect a cable between ports 1&2, and another cable between ports 3&4.
    2. Start this program: ./tloadbeep.

    Expected Results:  
    
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <math.h>

#include "../src/vpbapi.h"
#include "../src/verbose.h"

#define NUM_CH      4
#define NUM_BEEPERS 4
#define TX_N        400
#define RX_N        (TX_N/2)
#define AMP         5000
#define THRESH      (AMP/8)

// beeper states

#define STATE_TX 0
#define STATE_RX 1
#define PI       3.141
#define FS       8000.0
#define TIME_OUT (TX_N/1000+500)

void *channel_thread(void *pv);
void *play_sync(void *pv);
int tone_present(short buf[], int n, float f);
int vpb_echo_canc_force_adapt_on();
int vpb_echo_canc_force_adapt_off();
int vpb_comp_load(unsigned short board);
int kbhit();

int finito,event_finito;
int active_threads;
pthread_mutex_t mutex;

// state variables for each beeper

typedef struct {
	int    htx;       // channel handle of tx
	int    hrx;       // channel handle of rx
	float  freq;      // freq of tx tone
	int    delay;     // delay in cycle
	int    cycles;    // total cycles processed
	int    time_outs; // total no. time outs (errors, should be 0)
	short  tx_buf[3*TX_N];
	int    play_finished;
} BEEPER;

BEEPER beepers[] = {
	{0, 1,  500, 100, 0, 0},
	{1, 0, 1000, 150, 0, 0},
	{2, 3,  500, 175, 0 ,0},
	{3, 2, 1000, 200, 0 ,0},
};

int main() {
	pthread_t threads[NUM_CH];
        int i;

	active_threads = 0;
	pthread_mutex_init(&mutex, NULL);

	event_finito = finito = 0;
	verbose(1);

	// start a thread for each channel

	for(i=0; i<NUM_CH; i++) {
		vpb_open(1, i+1);
		vpb_sethook_sync(i,VPB_OFFHOOK);		 
	}

	for(i=0; i<NUM_BEEPERS; i++) {
		pthread_create(&threads[i], NULL, channel_thread, 
			       (void*)&beepers[i]);
	}

	printf("Press return to finish.....\n");

	//vpb_echo_canc_force_adapt_on();
	int its = 0;
	while(!kbhit()) {
		vpb_sleep(100);
		its++;
		if (its == 20) {
			its = 0;
			
			for(i=0; i<NUM_CH; i++) {
				printf("bp: %d, cycles %d time_outs %d\n", 
				       i, beepers[i].cycles,
				       beepers[i].time_outs);
			}
			vpb_comp_load(0);
			
		}
	}

	// signal all threads and wait for all to finish

	finito = 1;
	while(active_threads)
		vpb_sleep(10);

	for(i=0; i<NUM_CH; i++) {
		vpb_sethook_sync(i, VPB_ONHOOK);
		vpb_close(i);
	}

	pthread_mutex_destroy(&mutex);

	return 0;
}

void *channel_thread(void *pv)
{
	BEEPER     *b = (BEEPER*)pv;
	int        htx=b->htx;
	int        hrx=b->hrx;
	short      rx_buf[RX_N];
	int        i,k;
	int        tone;
	pthread_t  play_thread;

	// keep count of how many threads active

	pthread_mutex_lock(&mutex);
	active_threads++;
	pthread_mutex_unlock(&mutex);

	// init this port

	k = 0;
	for(i=0; i<TX_N; i++)
		b->tx_buf[k++] = 0;
	for(i=0; i<TX_N; i++)
		b->tx_buf[k++] = (short)(AMP*cos(2.0*PI*(float)i*b->freq/FS));
	for(i=0; i<TX_N; i++)
		b->tx_buf[k++] = 0;
	vpb_record_buf_start(hrx, VPB_LINEAR);
	vpb_play_buf_start(htx, VPB_LINEAR);

	while(!finito) {
		b->play_finished = 0;
		tone = 0;
		pthread_create(&play_thread, NULL, play_sync, (void*)b);
		do {
			vpb_record_buf_sync(hrx, (char*)rx_buf,sizeof(rx_buf));
			if(tone_present(rx_buf, RX_N, b->freq))
				tone = 1;
		} while (!b->play_finished);

		if (!tone)
			(b->time_outs)++;
		(b->cycles)++;	
		vpb_sleep(b->delay);
	}

	// decrement active thread count

	pthread_mutex_lock(&mutex);
	active_threads--;
	pthread_mutex_unlock(&mutex);

	return NULL;
}

void *play_sync(void *pv)
{
	BEEPER     *b = (BEEPER*)pv;
	int        htx=b->htx;

	if (htx == 0)
		printf("start play\n");
	vpb_play_buf_sync(htx, (char*)b->tx_buf, 3*TX_N*sizeof(short));
	if (htx == 0)
		printf("finish play\n");
	b->play_finished = 1;

	return NULL;
}

// returns 1 if tone of frequency f is present in buf of n samples, 
// otherwise returns 0

int tone_present(short buf[], int n, float f) {
	int i;
	float re,im,x;

	// single frequency discrete fourier transform
	x = 0.0;
	re = im = 0.0;
	for(i=0; i<n; i++) {
		re += (float)buf[i] * cos(2*PI*i*f/FS);
		im += (float)buf[i] * sin(2*PI*i*f/FS);
	}

	x = sqrt(re*re + im*im)/n;
	//printf("x = %f\n",x);

	if (x > THRESH)
		return 1;
	else
		return 0;
}


		
